"use client"

import { useState, useTransition } from 'react'
import cookies from 'js-cookie';
import { uploadAdmissionLetter } from '../utils/app'
import { IoMdClose } from 'react-icons/io'
import { notify } from '../utils/helper';

function UploadAdmissionLetter({ onClose }) {
    const [admLetter, setAdmLetter] = useState('');
    const [error, setError] = useState('');
    const [isPending, startTransition] = useTransition();

    const onFileChange = (e) => {

        const file = e.target.files[0]

        if (!file) {
            setError('No file selected');
            return;
        }

        const validTypes = ['image/png', 'image/jpeg', 'image/jpg', 'application/pdf'];
        const maxSize = 500 * 1024; // 500 KB in bytes

        if (!validTypes.includes(file.type)) {
            setError('Only PNG, JPG, JPEG and PDF files are allowed');
            e.target.value = '';
            return;
        }

        if (file.size > maxSize) {
            setError('File size must be less than 500 KB');
            e.target.value = '';
            return;
        }
        setAdmLetter(e.target.files[0])

    };

    const handleSubmit = async (e) => {
        e.preventDefault();
        try {
            const formData = new FormData();
            formData.append('adm_letter', admLetter);
            const token = cookies.get('aspirant');
            const res = await uploadAdmissionLetter(token, formData);

            if (!res) throw new Error('Failed to upload');
            notify('success', 'Successfully uploaded');
            onClose();
        } catch (error) {
            const errorMessage = error.response?.data?.message || error.message || String(error);
            notify('error', errorMessage);
        }
    }

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div className="bg-orange-400 rounded-lg overflow-hidden shadow-xl transform transition-all max-w-lg w-full">
                <div className="text-orange-900 px-4 py-5 sm:p-6">
                    <div className="flex justify-between items-center border-b border-orange-200 pb-5">
                        <h3 className="text-lg leading-6 font-medium text-orange-900">
                            Upload Admission Letter
                        </h3>
                        <IoMdClose
                            className="cursor-pointer hover:text-gray-600 transition ease-in-out duration-150"
                            onClick={onClose}
                            aria-label="Close modal"
                        />
                    </div>
                    <form onSubmit={(e) => startTransition(() => handleSubmit(e))} className="space-y-6 pt-6 pb-5">
                        <input
                            type="file"
                            name="adm_letter"
                            className="block w-full text-sm file:mr-4 file:py-2 file:px-4 file:rounded file:border-0 file:text-sm file:font-semibold file:bg-orange-500 file:text-orange-900 hover:file:bg-orange-600"
                            onChange={onFileChange}
                            accept="image/png, image/jpeg, image/jpg, application/pdf"
                            required
                        />
                        {<p className="text-center text-sm text-red-600">{error}</p>}
                        <div className="flex justify-center">
                            <button
                                type="submit"
                                className="inline-flex justify-center py-2 px-4 font-semibold border border-transparent shadow-sm text-sm rounded-md bg-orange-600 hover:bg-orange-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-orange-500"
                            >
                                {isPending ? 'Uploading...' : 'Upload'}
                            </button>

                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
}

export default UploadAdmissionLetter;
